require("dotenv").config();
const {
  app,
  BrowserWindow,
  ipcMain,
  dialog,
  protocol,
  shell,
} = require("electron");
const path = require("path");
const os = require("os");
const fs = require("fs");
const express = require("express");
const isDev = process.env.NODE_ENV === "development";
const { initialize } = require("@aptabase/electron/main");
const { spawn } = require("child_process");
initialize("A-EU-1883920536");
const server = express();
const PORT = 5678;
// Auto Updater
const autoUpdater = require("electron-updater");

// Serve static files from the dist directory
server.use(express.static(
  isDev ? path.join(__dirname, "../dist") : path.join(__dirname, "dist")
));

// Define your Express routes here
server.get("*", (req, res) => {
  res.sendFile(
    isDev ? path.join(__dirname, "../dist/index.html") : path.join(__dirname, "dist/index.html")
  );
});

// Start the Express server
server.listen(PORT, () => {
  // // console.log(`Express server running on http://localhost:${PORT}`);
});

let mainWindow;
function getFileExtension(filePath) {
  return path.extname(filePath).slice(1);
}
function createWindow() {
  if (!mainWindow) {
    mainWindow = new BrowserWindow({
      width: 1040,
      minWidth: 880,
      height: 880,
      minHeight: 720,
      maxHeight: 1800,
      maxWidth: 1240,
      icon: path.join(__dirname, "../dist/assets/codesnippets"),
      webPreferences: {
        nodeIntegration: false,
        contextIsolation: true,
        preload: path.join(
          __dirname,
          isDev ? "preload.js" : "../build/preload.js"
        ),
        devTools: true,
      },
    });

    // mainWindow.webContents.on("before-input-event", (event, input) => {
    //   if (
    //     input.key === "F5" ||
    //     (input.control && input.key === "r") ||
    //     (input.control && input.shift && input.key === "R") ||
    //     (input.control && input.shift && input.key === "I")
    //   ) {
    //     event.preventDefault();
    //   }
    // });
    mainWindow.loadURL("http://localhost:5678");

    // Check for updates
    if (typeof autoUpdater.checkForUpdatesAndNotify === 'function') autoUpdater.checkForUpdatesAndNotify();

    mainWindow.on("closed", () => {
      mainWindow = null;
    });
  }
}

ipcMain.handle("open-external", (event, url) => {
  return shell.openExternal(url);
});
ipcMain.handle("show-open-dialog", async (event, options) => {
  const result = await dialog.showOpenDialog(options);
  return result.filePaths[0];
});

ipcMain.handle("show-open-file-dialog", async (event, options) => {
  const result = await dialog.showOpenDialog(options);
  return result;
});

ipcMain.handle("check-file-exists", async (event, filePath) => {
  try {
    await fs.promises.access(filePath);
    return true;
  } catch {
    return false;
  }
});

ipcMain.handle("read-directory", async (event, dirPath) => {
  const entries = await fs.promises.readdir(dirPath, { withFileTypes: true });
  return entries.map((entry) => ({
    name: entry.name,
    isDirectory: entry.isDirectory(),
  }));
});

ipcMain.handle("read-file", async (event, filePath) => {
  const fileContent = await fs.promises.readFile(filePath, "utf-8");
  return fileContent;
});

ipcMain.handle("get-user-data-path", (event, fileName) => {
  return path.join(app.getPath("userData"), fileName);
});

// Ollama ipc
ipcMain.handle("start-ollama", async () => {
  return new Promise((resolve, reject) => {
    const process = spawn("ollama", ["serve"]);

    process.stdout.on("data", (data) => {
      // console.log(`Ollama output: ${data}`);
    });

    process.stderr.on("data", (data) => {
      console.error(`Ollama error: ${data}`);
    });

    process.on("close", (code) => {
      if (code === 0) {
        resolve("Ollama started successfully");
      } else {
        reject(new Error(`Ollama failed to start with code ${code}`));
      }
    });
  });
});

function handleProtocol(url) {
  const sendDeeplink = () => {
    if (mainWindow && mainWindow.webContents.isLoading()) {
      mainWindow.webContents.once("did-finish-load", () => {
        mainWindow.webContents.send("deeplink", url);
      });
    } else if (mainWindow) {
      mainWindow.webContents.send("deeplink", url);
    } else {
      createWindow();
      mainWindow.webContents.once("did-finish-load", () => {
        mainWindow.webContents.send("deeplink", url);
      });
    }
  };

  if (app.isReady()) {
    sendDeeplink();
  } else {
    app.on("ready", sendDeeplink);
  }
}

app.whenReady().then(() => {
  createWindow();

  app.on("activate", () => {
    if (BrowserWindow.getAllWindows().length === 0) createWindow();
  });

  // app.on("open-url", (event, url) => {
  //   event.preventDefault();
  //   handleProtocol(url);
  // });

  if (process.platform === "win32") {
    handleProtocol(process.argv.slice(1).join(" "));
  }
});

app.on("second-instance", (event, commandLine) => {
  if (process.platform === "win32") {
    handleProtocol(commandLine.slice(1).join(" "));
  }
  if (mainWindow) {
    if (mainWindow.isMinimized()) mainWindow.restore();
    mainWindow.focus();
  }
});

app.on("window-all-closed", () => {
  if (process.platform !== "darwin") {
    app.quit();
  }
});

const gotTheLock = app.requestSingleInstanceLock();

if (!gotTheLock) {
  app.quit();
} else {
  app.on("second-instance", (event, commandLine, workingDirectory) => {
    if (process.platform === "win32") {
      const deeplinkingUrl = commandLine.slice(1);
      handleProtocol(deeplinkingUrl);
    }
    if (mainWindow) {
      if (mainWindow.isMinimized()) mainWindow.restore();
      mainWindow.focus();
    }
  });
}

app.setAsDefaultProtocolClient("codesnippets");
if (process.platform === "win32") {
  const deeplinkingUrl = process.argv.slice(1);
  handleProtocol(deeplinkingUrl);
}
module.exports = { app, BrowserWindow };
